<?php
namespace EdgeCore\PostTypes\Portfolio\Shortcodes;

use EdgeCore\Lib;

/**
 * Class PortfolioList
 * @package EdgeCore\PostTypes\Portfolio\Shortcodes
 */
class PortfolioList implements Lib\ShortcodeInterface {
	/**
	 * @var string
	 */
	private $base;

	public function __construct() {
		$this->base = 'edgtf_portfolio_list';

		add_action('vc_before_init', array($this, 'vcMap'));
	}

	/**
	 * Returns base for shortcode
	 * @return string
	 */
	public function getBase() {
		return $this->base;
	}

	/**
	 * Maps shortcode to Visual Composer
	 *
	 * @see vc_map
	 */
	public function vcMap() {
		if(function_exists('vc_map')) {

			$icons_array= array();
			if(edge_cpt_theme_installed()) {
				$icons_array = \BlazeEdgeIconCollections::get_instance()->getVCParamsArray();
			}

			vc_map( array(
					'name' => esc_html__('Portfolio List','edge-cpt'),
					'base' => $this->getBase(),
					'category' => esc_html__('by EDGE','edge-cpt'),
					'icon' => 'icon-wpb-portfolio extended-custom-icon',
					'allowed_container_element' => 'vc_row',
					'params' => array(
						array(
							'type' => 'dropdown',
							'heading' => esc_html__('Portfolio List Template','edge-cpt'),
							'param_name' => 'type',
							'value' => array(
								esc_html__('Standard','edge-cpt') => 'standard',
								esc_html__('Gallery','edge-cpt') => 'gallery',
								esc_html__('Masonry','edge-cpt') => 'masonry',
								esc_html__('Pinterest','edge-cpt') => 'pinterest',
								esc_html__('Pinterest Thick','edge-cpt') => 'pinterest-thick',
							),
							'admin_label' => true
						),
						array(
							'type' => 'dropdown',
							'heading' => esc_html__('Space Between Columns','edge-cpt'),
							'param_name' => 'space_between',
							'value' => array(
								esc_html__('Medium','edge-cpt') => 'medium',
								esc_html__('Small','edge-cpt') => 'small',
								esc_html__('None','edge-cpt') => 'none',
							)
						),
						array(
							'type' => 'dropdown',
							'heading' => esc_html__('Hover Type','edge-cpt'),
							'param_name' => 'hover_type',
							'value' => array(
								esc_html__('Semi-transparent Overlay','edge-cpt') => 'semi-transparent-overlay',
								esc_html__('Dark Overlay','edge-cpt') => 'dark-overlay',
								esc_html__('Light Overlay','edge-cpt') => 'light-overlay',
								esc_html__('Zoom Lightbox','edge-cpt') => 'zoom-lightbox',
								esc_html__('Floating Title','edge-cpt') => 'floating-title',
								esc_html__('Strikethrough','edge-cpt') => 'strikethrough'
							),
							'save_always' => true,
							'dependency' => array('element' => 'type', 'value' => array('gallery','masonry','pinterest-thick'))
						),
						array(
							'type' => 'dropdown',
							'heading' => esc_html__('Title Tag','edge-cpt'),
							'param_name' => 'title_tag',
							'value' => array(
								''   => '',
								'h2' => 'h2',
								'h3' => 'h3',
								'h4' => 'h4',
								'h5' => 'h5',
								'h6' => 'h6',
							),
							'description' => esc_html__('Floating Title and Strikethrough hover will not be affected','edge-cpt')
						),
						array(
							'type' => 'dropdown',
							'heading' => esc_html__('Image Proportions','edge-cpt'),
							'param_name' => 'image_size',
							'value' => array(
								esc_html__('Original','edge-cpt') => 'full',
								esc_html__('Square','edge-cpt') => 'square',
								esc_html__('Landscape','edge-cpt') => 'landscape',
								esc_html__('Portrait','edge-cpt') => 'portrait'
							),
							'save_always' => true,
							'dependency' => array('element' => 'type', 'value' => array('standard', 'gallery'))
						),
						array(
							'type' => 'dropdown',
							'heading' => esc_html__('Show More','edge-cpt'),
							'param_name' => 'show_more',
							'value' => array(
								esc_html__('Load More Button','edge-cpt') => 'load_more',
								esc_html__('Infinite Scroll','edge-cpt') => 'infinite_scroll',
								esc_html__('None','edge-cpt') => 'none'
							),
							'save_always' => true,
							'description' => esc_html__('Default Value is load more','edge-cpt')
						),
						array(
							'type' => 'dropdown',
							'heading' => esc_html__('Order By','edge-cpt'),
							'param_name' => 'order_by',
							'value' => array(
								esc_html__('Date','edge-cpt') => 'date',
								esc_html__('Title','edge-cpt') => 'title',
								esc_html__('Menu Order','edge-cpt') => 'menu_order',
							),
							'save_always' => true,
							'group' => esc_html__('Query and Layout Options','edge-cpt')
						),
						array(
							'type' => 'dropdown',
							'heading' => esc_html__('Order','edge-cpt'),
							'param_name' => 'order',
							'value' => array(
								esc_html__('ASC','edge-cpt') => 'ASC',
								esc_html__('DESC','edge-cpt') => 'DESC',
							),
							'save_always' => true,
							'group' => esc_html__('Query and Layout Options','edge-cpt')
						),
						array(
							'type' => 'textfield',
							'heading' => esc_html__('One-Category Portfolio List','edge-cpt'),
							'param_name' => 'category',
							'description' => esc_html__('Enter one category slug (leave empty for showing all categories)','edge-cpt'),
							'group' => esc_html__('Query and Layout Options','edge-cpt')
						),
						array(
							'type' => 'textfield',
							'heading' => esc_html__('Number of Portfolios Per Page','edge-cpt'),
							'param_name' => 'number',
							'value' => '-1',
							'description' => esc_html__('(enter -1 to show all)','edge-cpt'),
							'group' => esc_html__('Query and Layout Options','edge-cpt')
						),
						array(
							'type' => 'dropdown',
							'heading' => esc_html__('Number of Columns','edge-cpt'),
							'param_name' => 'columns',
							'value' => array(
								'' => '',
								esc_html__('One','edge-cpt') => '1',
								esc_html__('Two','edge-cpt') => '2',
								esc_html__('Three','edge-cpt') => '3',
								esc_html__('Four','edge-cpt') => '4',
								esc_html__('Five','edge-cpt') => '5',
								esc_html__('Six','edge-cpt') => '6'
							),
							'admin_label' => true,
							'description' => esc_html__('Default value is Three','edge-cpt'),
							'dependency' => array('element' => 'type', 'value' => array('standard','gallery')),
							'group' => esc_html__('Query and Layout Options','edge-cpt')
						),
						array(
							'type' => 'dropdown',
							'heading' => esc_html__('Grid Size','edge-cpt'),
							'param_name' => 'grid_size',
							'value' => array(
								esc_html__('Default','edge-cpt') => '',
								esc_html__('3 Columns Grid','edge-cpt') => 'three',
								esc_html__('4 Columns Grid','edge-cpt') => 'four',
								esc_html__('5 Columns Grid','edge-cpt') => 'five'
							),
							'dependency' => array('element' => 'type', 'value' => array('pinterest','pinterest-thick')),
							'group' => esc_html__('Query and Layout Options','edge-cpt')
						),
						array(
							'type' => 'dropdown',
							'heading' => esc_html__('Number of Columns','edge-cpt'),
							'param_name' => 'masonry_columns',
							'value' => array(
								esc_html__('Default','edge-cpt') => '',
								esc_html__('3 Columns','edge-cpt') => 'three',
								esc_html__('4 Columns','edge-cpt') => 'four'
							),
							'dependency' => array('element' => 'type', 'value' => array('masonry')),
							'group' => esc_html__('Query and Layout Options','edge-cpt')
						),
						array(
							'type' => 'textfield',
							'heading' => esc_html__('Show Only Projects with Listed IDs','edge-cpt'),
							'param_name' => 'selected_projects',
							'description' => esc_html__('Delimit ID numbers by comma (leave empty for all)','edge-cpt'),
							'group' => esc_html__('Query and Layout Options','edge-cpt')
						),
						array(
							'type' => 'dropdown',
							'heading' => esc_html__('Enable Category Filter','edge-cpt'),
							'param_name' => 'filter',
							'value' => array(
								esc_html__('No','edge-cpt') => 'no',
								esc_html__('Yes','edge-cpt') => 'yes'
							),
							'save_always' => true,
							'description' => esc_html__('Default value is No','edge-cpt'),
							'group' => esc_html__('Query and Layout Options','edge-cpt')
						),
						array(
							'type' => 'dropdown',
							'heading' => esc_html__('Filter Order By','edge-cpt'),
							'param_name' => 'filter_order_by',
							'value' => array(
								esc_html__('Name','edge-cpt')  => 'name',
								esc_html__('Count','edge-cpt') => 'count',
								esc_html__('Id','edge-cpt')    => 'id',
								esc_html__('Slug','edge-cpt')  => 'slug'
							),
							'save_always' => true,
							'description' => esc_html__('Default value is Name','edge-cpt'),
							'dependency' => array('element' => 'filter', 'value' => array('yes')),
							'group' => esc_html__('Query and Layout Options','edge-cpt')
						),
						array(
							'type' => 'dropdown',
							'heading' => esc_html__('Filter Alignment','edge-cpt'),
							'param_name' => 'filter_alignment',
							'value' => array(
								esc_html__('Center','edge-cpt')  => 'center',
								esc_html__('Left','edge-cpt') => 'left',
								esc_html__('Right','edge-cpt')    => 'right',
							),
							'dependency' => array('element' => 'filter', 'value' => array('yes')),
							'group' => esc_html__('Query and Layout Options','edge-cpt')
						)
					)
				)
			);
		}
	}

	/**
	 * Renders shortcodes HTML
	 *
	 * @param $atts array of shortcode params
	 * @param $content string shortcode content
	 * @return string
	 */
	public function render($atts, $content = null) {

		$args = array(
			'type' => 'standard',
			'space_between' => 'medium',
			'columns' => '3',
			'grid_size' => 'three',
			'masonry_columns' => 'three',
			'image_size' => 'full',
			'hover_type' => 'semi-transparent-overlay',
			'order_by' => 'date',
			'order' => 'ASC',
			'number' => '-1',
			'filter' => 'no',
			'filter_order_by' => 'name',
			'filter_alignment' => 'center',
			'category' => '',
			'selected_projects' => '',
			'show_more' => 'load_more',
			'title_tag' => '',
			'next_page' => '',
			'portfolio_slider' => '',
			'portfolios_shown' => ''
		);

		$params = shortcode_atts($args, $atts);
		extract($params);

		$query_array = $this->getQueryArray($params);
		$query_results = new \WP_Query($query_array);
		$params['query_results'] = $query_results;
		$params['masonry_filter'] = '';
		$masonry_types = false;

		if ($type == 'masonry' || $type =='pinterest' || $type =='pinterest-thick') {
			$masonry_types = true;
		}
		$params['masonry_types'] = $masonry_types;

		if ($params['title_tag'] == '') {
			$params['title_tag'] = $this->getPortfolioTitleTag($hover_type);
		}

		$classes = $this->getPortfolioClasses($params);
		$data_atts = $this->getDataAtts($params);
		$data_atts .= 'data-max-num-pages = '.$query_results->max_num_pages;

		$html = '';

		if($filter == 'yes' && $masonry_types){
			$params['filter_categories'] = $this->getFilterCategories($params);
			$params['masonry_filter'] = 'edgtf-masonry-filter';

			if ($params['filter_alignment'] !== '') {
				$params['masonry_filter'] .= ' edgtf-ptf-filter-alignment-'.$params['filter_alignment'];
			}

			$html .= edge_cpt_get_shortcode_module_template_part('portfolio','portfolio-filter', '', $params);
		}

		$html .= '<div class = "edgtf-portfolio-list-holder-outer '.$classes.'" '.$data_atts. '>';

		if($filter == 'yes' && ($type == 'standard' || $type =='gallery')){
			$params['filter_categories'] = $this->getFilterCategories($params);

            if ($params['filter_alignment'] !== '') {
                $params['masonry_filter'] .= ' edgtf-ptf-filter-alignment-'.$params['filter_alignment'];
            }
			$html .= edge_cpt_get_shortcode_module_template_part('portfolio','portfolio-filter', '', $params);
		}

		$html .= '<div class = "edgtf-portfolio-list-holder clearfix" >';
		if($masonry_types){
			$html .= '<div class="edgtf-portfolio-list-masonry-grid-sizer"></div>';
			$html .= '<div class="edgtf-portfolio-list-masonry-grid-gutter"></div>';
		}

		if($query_results->have_posts()):
			while ( $query_results->have_posts() ) : $query_results->the_post();

				$params['current_id'] = get_the_ID();
				$params['thumb_size'] = $this->getImageSize($params);
				$params['category_html'] = $this->getItemCategoriesHtml($params);
				$params['categories'] = $this->getItemCategories($params);
				$params['article_masonry_size'] = $this->getMasonrySize($params);
				$params['item_link'] = $this->getItemLink($params);
				$params['hover_params']['category_html'] = $params['category_html'];
				$params['hover_params']['title_tag'] = $params['title_tag'];
				$params['hover_params']['item_link'] = $params['item_link'];

				$html .= edge_cpt_get_shortcode_module_template_part('portfolio',$type, '', $params);

			endwhile;
		else:

			$html .= '<p>'. esc_html__( 'Sorry, no posts matched your criteria.' ) .'</p>';

		endif;
		if(($type =='gallery' || $type == 'standard') && $portfolio_slider !== 'yes'){
			for($i=0;$i<(int)$columns;$i++){
				$html .= "<div class='edgtf-portfolio-gap'></div>\n";
			}
		}

		$html .= '</div>'; //close edgtf-portfolio-list-holder
		if($show_more !== 'none'){
			$html .= edge_cpt_get_shortcode_module_template_part('portfolio','load-more-template', '', $params);
		}
		wp_reset_postdata();
		$html .= '</div>'; // close edgtf-portfolio-list-holder-outer
		return $html;
	}

	/**
	 * Generates portfolio list query attribute array
	 *
	 * @param $params
	 *
	 * @return array
	 */
	public function getQueryArray($params){

		$query_array = array();

		$query_array = array(
			'post_type' => 'portfolio-item',
			'orderby' =>$params['order_by'],
			'order' => $params['order'],
			'posts_per_page' => $params['number']
		);

		if(!empty($params['category'])){
			$query_array['portfolio-category'] = $params['category'];
		}

		$project_ids = null;
		if (!empty($params['selected_projects'])) {
			$project_ids = explode(',', $params['selected_projects']);
			$query_array['post__in'] = $project_ids;
		}

		$paged = '';
		if(empty($params['next_page'])) {
			if(get_query_var('paged')) {
				$paged = get_query_var('paged');
			} elseif(get_query_var('page')) {
				$paged = get_query_var('page');
			}
		}

		if(!empty($params['next_page'])){
			$query_array['paged'] = $params['next_page'];

		}else{
			$query_array['paged'] = 1;
		}

		return $query_array;
	}

	/**
	 * Generates portfolio classes
	 *
	 * @param $params
	 *
	 * @return string
	 */
	public function getPortfolioClasses($params){
		$classes = array();
		$type = $params['type'];
		$spacing = $params['space_between'];
		$hover_type = $params['hover_type'];
		$columns = $params['columns'];
		$grid_size = $params['grid_size'];
		$masonry_columns = $params['masonry_columns'];

		switch($type):
			case 'standard':
				$classes[] = 'edgtf-ptf-standard';
				break;
			case 'gallery':
				$classes[] = 'edgtf-ptf-gallery';
				break;
			case 'masonry':
				$classes[] = 'edgtf-ptf-masonry';
				break;
			case 'pinterest':
				$classes[] = 'edgtf-ptf-pinterest';
				break;
			case 'pinterest-thick':
				$classes[] = 'edgtf-ptf-pinterest-thick';
				break;
		endswitch;

		switch($spacing):
			case 'none':
				$classes[] = 'edgtf-ptf-no-space';
				break;
			case 'small':
				$classes[] = 'edgtf-ptf-small-space';
				break;
			case 'medium':
				$classes[] = 'edgtf-ptf-medium-space';
				break;
			default:
				$classes[] = 'edgtf-ptf-no-space';
				break;
		endswitch;

		if ($hover_type !== '' && $type !== 'standard' && $type !== 'pinterest'){
			$classes[] = 'edgtf-ptf-hover-'.$hover_type;
		}

		if(empty($params['portfolio_slider'])){ // portfolio slider mustn't have this classes

			if($type == 'standard' || $type == 'gallery'){
				switch ($columns):
					case '1':
						$classes[] = 'edgtf-ptf-one-column';
						break;
					case '2':
						$classes[] = 'edgtf-ptf-two-columns';
						break;
					case '3':
						$classes[] = 'edgtf-ptf-three-columns';
						break;
					case '4':
						$classes[] = 'edgtf-ptf-four-columns';
						break;
					case '5':
						$classes[] = 'edgtf-ptf-five-columns';
						break;
					case '6':
						$classes[] = 'edgtf-ptf-six-columns';
						break;
				endswitch;
			}
			if($params['show_more'] == 'load_more'){
				$classes[] = 'edgtf-ptf-load-more edgtf-ptf-show-more';
			}
			else if($params['show_more'] == 'infinite_scroll'){
				$classes[] = 'edgtf-ptf-infinite-scroll edgtf-ptf-show-more';
			}

		}

		if($type == "pinterest" || $type == 'pinterest-thick'){
			switch ($grid_size):
				case 'three':
					$classes[] = 'edgtf-ptf-pinterest-three-columns';
					break;
				case 'four':
					$classes[] = 'edgtf-ptf-pinterest-four-columns';
					break;
				case 'five':
					$classes[] = 'edgtf-ptf-pinterest-five-columns';
					break;
			endswitch;
		}

		if($type == "masonry"){
			switch ($masonry_columns):
				case 'three':
					$classes[] = 'edgtf-ptf-masonry-three-columns';
					break;
				case 'four':
					$classes[] = 'edgtf-ptf-masonry-four-columns';
					break;
			endswitch;
		}
		if ($params['filter'] == 'yes') {
			$classes[] = 'edgtf-ptf-has-filter';

			if ($params['masonry_types']) {
				$classes[] = 'edgtf-ptf-masonry-filter';
			}
		}

		if(!empty($params['portfolio_slider']) && $params['portfolio_slider'] == 'yes'){
			$classes[] = 'edgtf-portfolio-slider-holder';
		}

		return implode(' ',$classes);

	}

	/**
	 * Generates portfolio title tag in dependency of hover type
	 *
	 * @param $hover_type
	 *
	 * @return string
	 */
	public function getPortfolioTitleTag($hover_type) {
		$title_tag = 'h3';

		switch ($hover_type) {
			case 'semi-transparent-overlay':
			case 'dark-overlay':
				$title_tag = 'h2';
				break;
			case 'light-overlay':
				$title_tag = 'h3';
				break;
			case 'floating-title':
				$title_tag = 'h5';
				break;
			case 'strikethrough':
				$title_tag = 'h2';
				break;
		}

		return $title_tag;
	}
	/**
	 * Generates portfolio image size
	 *
	 * @param $params
	 *
	 * @return string
	 */
	public function getImageSize($params){

		$thumb_size = 'full';
		$type = $params['type'];

		if($type == 'standard' || $type == 'gallery'){
			if(!empty($params['image_size'])){
				$image_size = $params['image_size'];

				switch ($image_size) {
					case 'landscape':
						$thumb_size = 'blaze_edge_landscape';
						break;
					case 'portrait':
						$thumb_size = 'blaze_edge_portrait';
						break;
					case 'square':
						$thumb_size = 'blaze_edge_square';
						break;
					case 'full':
						$thumb_size = 'full';
						break;
				}
			}
		}
		elseif($type == 'masonry'){

			$id = $params['current_id'];
			$masonry_size = get_post_meta($id, 'portfolio_masonry_dimenisions',true);

			switch($masonry_size):
				default :
					$thumb_size = 'blaze_edge_square';
					break;
				case 'large_width' :
					$thumb_size = 'blaze_edge_large_width';
					break;
				case 'large_height' :
					$thumb_size = 'blaze_edge_large_height';
					break;
				case 'large_width_height' :
					$thumb_size = 'blaze_edge_large_width_height';
					break;
			endswitch;
		}


		return $thumb_size;
	}
	/**
	 * Generates portfolio item categories ids.This function is used for filtering
	 *
	 * @param $params
	 *
	 * @return array
	 */
	public function getItemCategories($params){
		$id = $params['current_id'];
		$category_return_array = array();

		$categories = wp_get_post_terms($id, 'portfolio-category');

		foreach($categories as $cat){
			$category_return_array[] = 'portfolio_category_'.$cat->term_id;
		}
		return implode(' ', $category_return_array);
	}

	/**
	 * Generates portfolio item categories html based on id
	 *
	 * @param $params
	 *
	 * @return html
	 */
	public function getItemCategoriesHtml($params){
		$id = $params['current_id'];

		$categories = wp_get_post_terms($id, 'portfolio-category');
		$category_html = '<div class="edgtf-ptf-category-holder">';
		$k = 1;
		$hover_with_text = false;

		if ($params['hover_type'] == 'semi-transparent-overlay' || $params['hover_type'] == 'dark-overlay') {
			$hover_with_text = true;
		}

		if (is_array($categories) && count($categories) && $params['type'] !== 'standard' && $params['type'] !== 'pinterest'){
			if ($hover_with_text) {
				$category_html .= '<span class="edgtf-ptf-cat-text">'.esc_html__('category: ','edge-cpt').'</span>';
			} else {
				$category_html .= '<span>- </span>';
			}
		}

		foreach ($categories as $cat) {
			$category_html .= '<span>'.$cat->name.'</span>';
			if (count($categories) != $k) {
				$category_html .= ' / ';
			}
			$k++;
		}
		$category_html .= '</div>';
		return $category_html;
	}

	/**
	 * Generates masonry size class for each article( based on id)
	 *
	 * @param $params
	 *
	 * @return string
	 */
	public function getMasonrySize($params){
		$masonry_size_class = '';

		if($params['type'] == 'masonry'){

			$id = $params['current_id'];
			$masonry_size = get_post_meta($id, 'portfolio_masonry_dimenisions',true);
			switch($masonry_size):
				default :
					$masonry_size_class = 'edgtf-default-masonry-item';
					break;
				case 'large_width' :
					$masonry_size_class = 'edgtf-large-width-masonry-item';
					break;
				case 'large_height' :
					$masonry_size_class = 'edgtf-large-height-masonry-item';
					break;
				case 'large_width_height' :
					$masonry_size_class = 'edgtf-large-width-height-masonry-item';
					break;
			endswitch;
		}

		return $masonry_size_class;
	}
	/**
	 * Generates filter categories array
	 *
	 * @param $params
	 *
	 * * @return array
	 */
	public function getFilterCategories($params){

		$cat_id = 0;
		$top_category = '';

		if(!empty($params['category'])){

			$top_category = get_term_by('slug', $params['category'], 'portfolio-category');
			if(isset($top_category->term_id)){
				$cat_id = $top_category->term_id;
			}

		}

		$args = array(
			'taxonomy' => 'portfolio-category',
			'child_of' => $cat_id,
			'order_by' => $params['filter_order_by']
		);

		$filter_categories = get_terms($args);

		return $filter_categories;

	}
	/**
	 * Generates datta attributes array
	 *
	 * @param $params
	 *
	 * @return array
	 */
	public function getDataAtts($params){

		$data_attr = array();
		$data_return_string = '';

		if(get_query_var('paged')) {
			$paged = get_query_var('paged');
		} elseif(get_query_var('page')) {
			$paged = get_query_var('page');
		} else {
			$paged = 1;
		}

		if(!empty($paged)) {
			$data_attr['data-next-page'] = $paged+1;
		}
		if(!empty($params['type'])){
			$data_attr['data-type'] = $params['type'];
		}
		if(!empty($params['columns'])){
			$data_attr['data-columns'] = $params['columns'];
		}
		if(!empty($params['grid_size'])){
			$data_attr['data-grid-size'] = $params['grid_size'];
		}
		if(!empty($params['order_by'])){
			$data_attr['data-order-by'] = $params['order_by'];
		}
		if(!empty($params['order'])){
			$data_attr['data-order'] = $params['order'];
		}
		if(!empty($params['number'])){
			$data_attr['data-number'] = $params['number'];
		}
		if(!empty($params['image_size'])){
			$data_attr['data-image-size'] = $params['image_size'];
		}
		if(!empty($params['hover_type'])){
			$data_attr['data-hover-type'] = $params['hover_type'];
		}
		if(!empty($params['filter'])){
			$data_attr['data-filter'] = $params['filter'];
		}
		if(!empty($params['filter_order_by'])){
			$data_attr['data-filter-order-by'] = $params['filter_order_by'];
		}
		if(!empty($params['category'])){
			$data_attr['data-category'] = $params['category'];
		}
		if(!empty($params['selected_projects'])){
			$data_attr['data-selected-projects'] = $params['selected_projects'];
		}
		if(!empty($params['show_more'])){
			$data_attr['data-show-more'] = $params['show_more'];
		}
		if(!empty($params['title_tag'])){
			$data_attr['data-title-tag'] = $params['title_tag'];
		}
		if(!empty($params['portfolio_slider']) && $params['portfolio_slider']=='yes'){
			$data_attr['data-items'] = $params['portfolios_shown'];
		}

		foreach($data_attr as $key => $value) {
			if($key !== '') {
				$data_return_string .= $key . '= "' . esc_attr( $value ) . '" ';
			}
		}
		return $data_return_string;
	}

	public function getItemLink($params){

		$id = $params['current_id'];
		$portfolio_link = get_permalink($id);
		if (get_post_meta($id, 'portfolio_external_link',true) !== ''){
			$portfolio_link = get_post_meta($id, 'portfolio_external_link',true);
		}

		return $portfolio_link;

	}
}